package com.android.pingpong;

import com.android.pingpong.objects.*;
import com.android.pingpong.R;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.graphics.Paint.Style;
import android.preference.PreferenceManager;
import android.view.KeyEvent;
import android.view.SurfaceHolder;

public class GameManager extends Thread
{
    private interface DrawHelper
    {
        void draw(Canvas canvas);
    }

    private static final int LOSE_PAUSE = 2000;
    private static final int FIELD_WIDTH = 300;
    private static final int FIELD_HEIGHT = 250;

    /** ,     */
    private SurfaceHolder mSurfaceHolder;
    
    /**   (  . ,     ,  ) */
    private boolean mRunning;
    
    /**   */
    private Paint mPaint;
    private Paint mScorePaint;
    private Paint mPausePaint;
    
    /**    */
    private Rect mField;
    
    /**  */
    private Ball mBall;    
    
    /** ,   */
    private Racquet mUs;
    
    /** ,  */
    private Racquet mThem;

    /**  */
    private Bitmap mBackground;
    
    /**       */
    private boolean mInitialized;

    /**   ,     */
    private static int mMaxScore = 3;

    public static void setMaxScore(int value)
    {
        mMaxScore = value;
    }
    
    /**      */
    private boolean mPaused;

    /**     */
    private DrawHelper mDrawScreen;
    
    /**    */
    private DrawHelper mDrawPause;
    
    /**     */
    private DrawHelper mDrawGameover;
    
    /**
     * 
     * @param surfaceHolder  
     * @param context  
     */
    public GameManager(SurfaceHolder surfaceHolder, Context context)
    {
        mSurfaceHolder = surfaceHolder;
        Resources res = context.getResources();
        mRunning = false;
        mInitialized = false;

        //     
        mPaint = new Paint();
        mPaint.setColor(Color.BLUE);
        mPaint.setStrokeWidth(2);
        mPaint.setStyle(Style.STROKE);
        
        //    
        mScorePaint = new Paint();
        mScorePaint.setTextSize(20);
        mScorePaint.setStrokeWidth(1);
        mScorePaint.setStyle(Style.FILL);
        mScorePaint.setTextAlign(Paint.Align.CENTER);
        mScorePaint.setTypeface(Typeface.createFromAsset(context.getAssets(), "fonts/Mini.ttf"));

        //    
        mPausePaint = new Paint();
        mPausePaint.setStyle(Style.FILL);
        mPausePaint.setColor(Color.argb(100, 50, 50, 80));

        //  
        mField = new Rect();
        mBall = new Ball(res.getDrawable(R.drawable.ball));
        mUs = new Racquet(res.getDrawable(R.drawable.us));
        mThem = new Racquet(res.getDrawable(R.drawable.them));

        //  
        SharedPreferences settings = PreferenceManager.getDefaultSharedPreferences(context);

        String difficulty = settings.getString(res.getString(R.string.pref_difficulty), res.getString(R.string.difficulty_normal));        
        setDifficulty(difficulty);
        
        int maxScore = Integer.parseInt(settings.getString(res.getString(R.string.pref_max_score), "10"));
        setMaxScore(maxScore);
        
        //    
        mDrawScreen = new DrawHelper()
        {
            public void draw(Canvas canvas)
            {
                refreshCanvas(canvas);
            }
        };
    
        //    
        mDrawPause = new DrawHelper()
        {
            public void draw(Canvas canvas)
            {
                canvas.drawRect(mField, mPausePaint);
            }
        };
        
        //     
        mDrawGameover = new DrawHelper()
        {
            public void draw(Canvas canvas)
            {
                //    
                refreshCanvas(canvas);
                
                // ,      
                String message = "";
                if (mUs.getScore() > mThem.getScore())
                {
                    mScorePaint.setColor(Color.GREEN);
                    message = "You won";
                }
                else
                {
                    mScorePaint.setColor(Color.RED);
                    message = "You lost";
                }
                mScorePaint.setTextSize(30);
                canvas.drawText(message, mField.centerX(), mField.centerY(), mScorePaint);                
            }
        };
    }
    
    private void setDifficulty(String difficulty)
    {
        if (difficulty == "Easy")
        {
            mBall.setSpeed(1);
            mUs.setSpeed(5);
            mThem.setSpeed(5);
            
        }
        else if (difficulty == "Normal")
        {
            mBall.setSpeed(2);
            mUs.setSpeed(3);
            mThem.setSpeed(3);
        }
        else if (difficulty == "Hard")
        {
            mBall.setSpeed(3);
            mUs.setSpeed(1);
            mThem.setSpeed(1);
        }        
    }
    
    /** 
     *   
     * @param helper     
     */
    private void draw(DrawHelper helper)
    {
        Canvas canvas = null;
        try
        {
            //  Canvas-
            canvas = mSurfaceHolder.lockCanvas(); 
            synchronized (mSurfaceHolder)
            {
                if (mInitialized)
                {
                    helper.draw(canvas);
                }
            }
        }
        catch (Exception e) { }
        finally
        {
            if (canvas != null)
            {
                mSurfaceHolder.unlockCanvasAndPost(canvas);
            }
        }
    }
    
    /**
     *   
     * @param running
     */
    public void setRunning(boolean running)
    {
        mRunning = running;
    }
    
    @Override
    /** ,    */
    public void run()
    {
        while (mRunning)
        {
            if (mPaused) continue;
            
            if (mInitialized)
            {
                updateObjects(); //  
                draw(mDrawScreen);
                try
                {
                    sleep(20);
                }
                catch (InterruptedException e) {};
            }
        }
        draw(mDrawGameover);
    }
    
    /**     */
    private void refreshCanvas(Canvas canvas)
    {
        //   
        canvas.drawBitmap(mBackground, 0, 0, null);
        
        //   
        canvas.drawRect(mField, mPaint);
        
        //   
        mBall.draw(canvas);
        mUs.draw(canvas);
        mThem.draw(canvas);

        //  
        mScorePaint.setColor(Color.RED);
        canvas.drawText(String.valueOf(mThem.getScore()), mField.centerX(), mField.top - 10, mScorePaint);
        mScorePaint.setColor(Color.GREEN);
        canvas.drawText(String.valueOf(mUs.getScore()), mField.centerX(), mField.bottom + 25, mScorePaint);
    }

    /**     */
    private void updateObjects()
    {
        mBall.update();
        mUs.update();
        moveAI();
        
        //       
        placeInBounds(mUs);
        placeInBounds(mThem);
        
        //     
        if (mBall.getLeft() <= mField.left)
        {
            mBall.setLeft(mField.left + Math.abs(mField.left - mBall.getLeft()));
            mBall.reflectVertical();
        }
        else if (mBall.getRight() >= mField.right)
        {
            mBall.setRight(mField.right - Math.abs(mField.right - mBall.getRight()));
            mBall.reflectVertical();
        }

        //     
        if (GameObject.intersects(mBall, mUs))
        {
            mBall.setBottom(mUs.getBottom() - Math.abs(mUs.getBottom() - mBall.getBottom()));
            mBall.reflectHorizontal();
        }
        else if (GameObject.intersects(mBall, mThem))
        {
            mBall.setTop(mThem.getTop() + Math.abs(mThem.getTop() - mBall.getTop()));
            mBall.reflectHorizontal();
        }
        
        //  
        if (mBall.getTop() <= mField.top)
        {
            mUs.incScore();
            reset();
        }
        
        if (mBall.getBottom() >= mField.bottom)
        {
            mThem.incScore();
            reset();
        }
        
        //   
        if (mUs.getScore() == mMaxScore || mThem.getScore() == mMaxScore)
        {
            this.mRunning = false;
        }
    }
    
    /**   */
    private void moveAI()
    {
        if (mThem.getLeft() > mBall.getRight())
            mThem.setDirection(GameObject.DIR_LEFT);
        else if (mThem.getRight() < mBall.getLeft())
            mThem.setDirection(GameObject.DIR_RIGHT);
        else 
            mThem.setDirection(GameObject.DIR_NONE);
        mThem.update();
    }

    /**
     *   ,     
     * @param screenHeight  
     * @param screenWidth  
     */
    public void initPositions(int screenHeight, int screenWidth)
    {
        mInitialized = false;

        int left = (screenWidth - FIELD_WIDTH) / 2;
        int top = (screenHeight - FIELD_HEIGHT) / 2;
        
        mField.set(left, top, left + FIELD_WIDTH, top + FIELD_HEIGHT);
        
        mBackground = Bitmap.createBitmap(screenWidth, screenHeight, Bitmap.Config.RGB_565);
        
        //     
        mBall.setCenterX(mField.centerX());
        mBall.setCenterY(mField.centerY());
    
        //   -   
        mUs.setCenterX(mField.centerX());
        mUs.setBottom(mField.bottom);
    
        //   -   
        mThem.setCenterX(mField.centerX());
        mThem.setTop(mField.top);
        
        mInitialized = true;
    }
    
    /**
     *   
     * @param keyCode   
     * @return    
     */
    public boolean doKeyDown(int keyCode)
    {
        switch (keyCode)
        {
            case KeyEvent.KEYCODE_DPAD_LEFT:
            case KeyEvent.KEYCODE_A:
                mUs.setDirection(GameObject.DIR_LEFT);
                return true;
            case KeyEvent.KEYCODE_DPAD_RIGHT:
            case KeyEvent.KEYCODE_D:
                mUs.setDirection(GameObject.DIR_RIGHT);
                return true;
            case KeyEvent.KEYCODE_DPAD_CENTER:
                mPaused = !mPaused;
                
                if (mPaused)
                    draw(mDrawPause);
                return true;
            default:
                return false;
        }   
    }
    /**
     *   
     * @param keyCode  
     * @return    
     */
    
    public boolean doKeyUp(int keyCode)
    {
        if (keyCode == KeyEvent.KEYCODE_DPAD_LEFT ||
            keyCode == KeyEvent.KEYCODE_A ||
            keyCode == KeyEvent.KEYCODE_DPAD_RIGHT ||
            keyCode == KeyEvent.KEYCODE_D
            )
        {
            mUs.setDirection(GameObject.DIR_NONE);
            return true;
        }
        return false;
    }
    
    /**
     *       ,     
     * @param r 
     */
    private void placeInBounds(Racquet r)
    {
        if (r.getLeft() < mField.left)
            r.setLeft(mField.left);
        else if (r.getRight() > mField.right)
            r.setRight(mField.right);
    }
    
    /**
     *       ,   LOSE_PAUSE
     */
    private void reset()
    {
        //    
        mBall.setCenterX(mField.centerX());
        mBall.setCenterY(mField.centerY());
        //     
        mBall.resetAngle();
        
        //    
        mUs.setCenterX(mField.centerX());
        mThem.setCenterX(mField.centerX());
        
        //  
        try
        {
            sleep(LOSE_PAUSE);
        }
        catch (InterruptedException iex)
        {
        }
    }
}
